<?php
// process_payment.php
require_once 'config.php'; // Pour accéder aux constantes

session_start();

function redirectToFormWithError($message, $oldInput = []) {
    $_SESSION['error'] = $message;
    $_SESSION['old_input'] = $oldInput;
    header('Location: index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    redirectToFormWithError('Invalid request method.');
}

// 1. Validation des données
// NOUVEAU : Utiliser la constante FLUXSENDS_IDENTIFIER de config.php
$fluxsendsIdentifier = FLUXSENDS_IDENTIFIER;

$firstName   = trim($_POST['first_name'] ?? '');
$lastName    = trim($_POST['last_name'] ?? '');
$email       = trim($_POST['email'] ?? '');
$mobile      = trim($_POST['mobile'] ?? '');
$amount      = trim($_POST['amount'] ?? '');
$currency    = trim($_POST['currency'] ?? '');
$details     = trim($_POST['details'] ?? '');

$oldInput = [
    // 'fluxsends_identifier' => $fluxsendsIdentifier, // Plus besoin d'ajouter à old_input car il vient de config
    'first_name' => $firstName,
    'last_name'  => $lastName,
    'email'      => $email,
    'mobile'     => $mobile,
    'amount'     => $amount,
    'currency'   => $currency,
    'details'    => $details,
];

// La validation de l'identifiant n'est plus nécessaire ici car il est une constante définie.
// Si empty($fluxsendsIdentifier) est vrai, c'est que la constante n'est pas définie dans config.php
if (empty($firstName) || empty($lastName) || empty($email) || empty($mobile) || empty($amount) || empty($currency) || empty($details)) {
    redirectToFormWithError('All fields are required (except identifier, which is pre-configured).', $oldInput);
}
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    redirectToFormWithError('Invalid email format.', $oldInput);
}
if (!is_numeric($amount) || (float)$amount <= 0) {
    redirectToFormWithError('Amount must be a positive number.', $oldInput);
}
if (!in_array($currency, $supportedCurrencies)) {
    redirectToFormWithError('Unsupported currency.', $oldInput);
}

$transactionReference = 'FLUX_TXN_' . uniqid() . '_' . time();

// Préparer les paramètres pour FluxSends
$parameters = [
    'identifier'      => $fluxsendsIdentifier, // UTILISER LA CONSTANTE
    'currency'        => $currency,
    'amount'          => (float)$amount,
    'gateway_methods' => [],
    'details'         => $details,
    'ipn_url'         => APP_BASE_URL . '/fluxsends_webhook.php',
    'cancel_url'      => APP_BASE_URL . '/payment_failure.php?ref=' . $transactionReference,
    'success_url'     => APP_BASE_URL . '/payment_success.php?ref=' . $transactionReference,
    'public_key'      => FLUXSENDS_PUBLIC_KEY,
    'site_name'       => SITE_NAME,
    'site_logo'       => SITE_LOGO_URL,
    'checkout_theme'  => CHECKOUT_THEME,
    'customer'        => [
        'first_name' => $firstName,
        'last_name'  => $lastName,
        'email'      => $email,
        'mobile'     => $mobile,
    ],
    'shipping_info'   => [
        'address_one' => '', 'address_two' => '', 'area' => '', 'city' => '',
        'sub_city'    => '', 'state' => '', 'postcode' => '', 'country' => '', 'others' => '',
    ],
    'billing_info'    => [
        'address_one' => '', 'address_two' => '', 'area' => '', 'city' => '',
        'sub_city'    => '', 'state' => '', 'postcode' => '', 'country' => '', 'others' => '',
    ]
];

$postData = http_build_query($parameters);

$ch = curl_init();
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_URL, FLUXSENDS_INITIATE_ENDPOINT);
curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Accept: application/json',
]);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

$result = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curlError = curl_error($ch);
curl_close($ch);

if ($curlError) {
    redirectToFormWithError('cURL Error: ' . $curlError, $oldInput);
}

$responseData = json_decode($result, true);

if ($httpCode >= 200 && $httpCode < 300) {
    if (isset($responseData['status']) && $responseData['status'] === 'success') {
        if (isset($responseData['redirect_url']) && !empty($responseData['redirect_url'])) {
            header('Location: ' . $responseData['redirect_url']);
            exit;
        } else {
            redirectToFormWithError('FluxSends success but no redirect URL provided. Raw response: ' . $result, $oldInput);
        }
    } else if (isset($responseData['status']) && $responseData['status'] === 'error') {
        $errorMessage = 'FluxSends API Error: ';
        if (isset($responseData['message']) && is_array($responseData['message'])) {
            $errorMessage .= implode(', ', $responseData['message']);
        } else if (isset($responseData['message'])) {
            $errorMessage .= $responseData['message'];
        } else {
            $errorMessage .= 'Unknown error. Raw response: ' . $result;
        }
        redirectToFormWithError($errorMessage, $oldInput);
    } else {
        redirectToFormWithError('Unexpected response from FluxSends. Raw response: ' . $result, $oldInput);
    }
} else {
    $errorMessage = 'HTTP Error from FluxSends: ' . $httpCode . '. ';
    if (isset($responseData['message']) && is_array($responseData['message'])) {
        $errorMessage .= implode(', ', $responseData['message']);
    } else if (isset($responseData['message'])) {
        $errorMessage .= $responseData['message'];
    } else {
        $errorMessage .= 'Response: ' . $result;
    }
    redirectToFormWithError($errorMessage, $oldInput);
}
?>